import pygame
from pygame.locals import *

from gummworld2 import State, View
from gummworld2 import data

import settings


class TextReader(pygame.sprite.Group):
    
    def __init__(self, text, rect):
        pygame.sprite.Group.__init__(self)
        if isinstance(text, str):
            text = text.split('\n')
        self.view = View(subsurface_rect=rect)
        self.rect = self.view.rect.inflate(-10,0)
        font_file = data.filepath('font', settings.textreader_font)
        self.font = pygame.font.Font(font_file, 16)
        self.fgcolor = Color('yellow')
        line_size = self.font.get_linesize()
        line_num = 0
        # blank image for paragraph separators
        help_on_help_image = self.font.render(
            '[Use arrow keys, Escape or mouse]',
            True, self.fgcolor)
        def make_sprite(image):
            s = pygame.sprite.Sprite()
            s.image = image
            s.rect = image.get_rect()
            s.rect.top = line_num*line_size
            s.rect.left = self.rect.left
            return s
        self.top_text = make_sprite(help_on_help_image)
        self.top_text.rect.centerx = self.view.rect.centerx
        self.top_line = line_num
        self.set_top_line = line_num
        self.add(self.top_text)
        line_num += 2
        for paragraph in tuple(text):
            for line in self._wrap_text(paragraph):
                line = line.rstrip()
                s = make_sprite(
                    self.font.render(line, True, self.fgcolor))
                self.add(s)
                line_num += 1
        line_num += 1
        bottom_text = make_sprite(help_on_help_image)
        bottom_text.rect.centerx = self.view.rect.centerx
        self.add(bottom_text)
        line_num += 1
        self.bottom_text = make_sprite(pygame.surface.Surface((0,0)))
        self.add(self.bottom_text)
    
    def scroll(self, adjust):
        """scroll text one line
        adjust -> 1=up; -1=down
        """
        if self.top_line != self.set_top_line:
            return adjust
        elif adjust < 0:
            # are we at the bottom?
            if self.rect.colliderect(self.bottom_text.rect):
                pass
            else:
                self.top_line -= 1
                return adjust + 1
        elif adjust > 0:
            # are we at the top?
            if self.rect.colliderect(self.top_text.rect):
                pass
            else:
                self.top_line += 1
                return adjust - 1
        return 0
    
    def update(self, *args):
        for n in range(settings.textreader_speed):
            if self.top_line == self.set_top_line:
                pass
            else:
                if self.top_line < self.set_top_line:
                    adjust = -1
                elif self.top_line > self.set_top_line:
                    adjust = 1
                for s in self.sprites():
                    s.rect.top += adjust
                    if s.rect.top == self.rect.top:
                        self.top_line = self.set_top_line
    
    def draw(self):
        self.view.clear()
        pygame.sprite.Group.draw(self, self.view.surface)
        pygame.draw.rect(self.view.surface, Color('grey'), self.view.rect, 1)
    
    def _wrap_text(self, text):
        font = self.font
        wrapped = []
        line = ''
        w_screen = self.rect.width
        w_line = 0
        w_space = font.size(' ')[0]
        for part in text.split(' '):
            w_part = font.size(part)[0]
            w_new = w_line+w_space+w_part
            if w_new > w_screen:
                ## line+part is too wide for screen; save line; start new
                ## line with part
                wrapped.append(line)
                line = part
                w_line = font.size(line)[0]
            elif w_part > w_screen:
                ## part is too wide for screen; break it up
                # save line if already processing one
                if w_line:
                    wrapped.append(line)
                # start checking entire width of part, reducing by one until
                # it fits on screen
                end = len(part)
                while w_part > w_screen:
                    # find largest slice of part that fits on screen
                    w_slice = w_screen + 1
                    while w_slice > w_screen:
                        end -= 1
                        slice_ = part[0:end]
                        w_slice = font.size(slice_)[0]
                    # save slice; remove slice from part; set up next loop
                    wrapped.append(slice_)
                    part = part[end:]
                    w_part = font.size(part)[0]
                    end = len(part)
                # if any part left, save it
                if len(part):
                    wrapped.append[part]
                # start new blank line
                line = ''
                w_line = 0
            else:
                ## line+part fits on screen; set up next loop
                line = ' '.join((line,part))
                w_line = w_new
        if len(line):
            wrapped.append(line)
        return wrapped
