import pygame
from pygame.locals import *

from gummworld2 import State, Vec2d
from gummworld2 import data, geometry, toolkit


## Finite state machine for letter buttons
class _BaseState:
    @staticmethod
    def enter(self):
        pass
    @staticmethod
    def exit(self):
        pass
    @staticmethod
    def think(self, dt):
        pass
    @staticmethod
    def claim(self, move_to):
        pass
    @staticmethod
    def free(self, move_to):
        pass
    @staticmethod
    def shuffle(self, move_to):
        pass

class FreeState(_BaseState):
    @staticmethod
    def claim(self, move_to):
        self.move_to = move_to
        self.pending_state = ClaimingState
    @staticmethod
    def shuffle(self, move_to):
        self.move_to = move_to
        self.pending_state = ShufflingState
    @staticmethod
    def think(self, dt):
        return self.pending_state
    @staticmethod
    def exit(self):
        self.pending_state = None

class ClaimingState(_BaseState):
    @staticmethod
    def free(self, move_to):
        self.move_to = move_to
        self.pending_state = FreeingState
    @staticmethod
    def think(self, dt):
        self.move(dt)
        if self.position == self.move_to:
            return ClaimedState

class ClaimedState(_BaseState):
    @staticmethod
    def free(self, move_to):
        self.move_to = move_to
        self.pending_state = FreeingState
    @staticmethod
    def think(self, dt):
        return self.pending_state
    @staticmethod
    def exit(self):
        self.pending_state = None

class FreeingState(_BaseState):
    @staticmethod
    def claim(self, move_to):
        self.move_to = move_to
        self.pending_state = ClaimingState
    @staticmethod
    def think(self, dt):
        self.move(dt)
        if self.position == self.move_to:
            return FreeState

class ShufflingState(_BaseState):
    @staticmethod
    def think(self, dt):
        self.move(dt)
        if self.position == self.move_to:
            return FreeState


class LetterButton(object):
    
    image = None
    
    def __init__(self, idx, char, char_image, parent_view):
        self.idx = idx
        self.char = char
        self.char_image = char_image
        self.char_rect = char_image.get_rect()
        self.parent = parent_view
        
        if not self.image:
            load = pygame.image.load
            file = data.filepath
            self.image = load(file('button','WoodDisplayAf2.gif'))
        self.rect = self.image.get_rect()
        
        self._position = self.rect.center
        self.step = Vec2d(0,0)
        self.move_to = self._position
        self.speed = 250  # pixels per second
        
        self.clicked = False
        self.shuffled = False
        
        self.pending_state = None
        self.current_state = _BaseState
        self.set_state(FreeState)
    
    @property
    def position(self):
        return self._position
    @position.setter
    def position(self, val):
        self._position = val
        x,y = val
        self.rect.center = round(x),round(y)
    
    def set_state(self, new_state):
        if new_state is not None and new_state != self.current_state:
            if __debug__: print '%s: %s -> %s' % (
                self, self.current_state.__name__, new_state.__name__
            )
            self.current_state.exit(self)
            self.current_state = new_state
            self.current_state.enter(self)
    
    def think(self, dt):
        new_state = self.current_state.think(self, dt)
        if new_state:
            self.set_state(new_state)
    
    def click(self, mouse_pos):
        pos = Vec2d(mouse_pos) - self.parent.abs_offset
        return self.rect.collidepoint(pos)
    
    def claim(self, move_to):
        self.current_state.claim(self, move_to)
    
    def free(self, move_to):
        self.current_state.free(self, move_to)
    
    def shuffle(self, move_to):
        self.current_state.shuffle(self, move_to)
    
    def move(self, dt):
        start,end = self.position, self.move_to
        if start == end:
            return
        angle = geometry.angle_of(start, end)
        speed = self.speed * dt
        if self.current_state == ShufflingState:
            speed *= .5
        step = Vec2d(geometry.point_on_circumference(start, speed, angle))
        dist = geometry.distance(start, end)
        if dist <= speed:
            self.step[:] = 0,0
            self.position = end
        else:
            self.step = Vec2d(step-self.position)
            self.position = step
    
    def draw(self):
        x,y = toolkit.interpolated_step(
            self.position, self.step, State.clock.interpolate)
        blit = self.parent.blit
        r = self.image.get_rect(center=(x,y))
        blit(self.image, r)
#        self.char_rect.center = self.rect.center
        r = self.char_image.get_rect(center=(x,y))
        blit(self.char_image, r)
    
    def __str__(self):
        return '%s(%s)' % (self.__class__.__name__,self.char)


if __name__ == '__main__':
    from gummworld2 import View
    import settings
    screen = pygame.display.set_mode((400,400))
    view = View(screen, Rect(100,100,200,200))
    clock = pygame.time.Clock()
    font_size = 22
    font_name = data.filepath('font',settings.letters_font_name)
    font = pygame.font.Font(font_name, font_size)
    button = LetterButton('a', font.render('a', True, Color('white')), view)
    poles = {FreeState:(150,150), ClaimedState:(50,50)}
    while 1:
        dt = clock.tick(30) / 1000.
        for e in pygame.event.get():
            if e.type == MOUSEBUTTONDOWN:
                if button.click(e.pos):
                    button.move_to = poles.get(button.current_state,(0,0))
        screen.fill((0,0,0))
        button.think(dt)
        button.draw(view)
        pygame.display.flip()
