import random

import pygame
from pygame.locals import *

from gummworld2 import State, View, Vec2d
from gummworld2 import data

import settings, dictionary
from letterbutton import LetterButton
from actionbutton import ActionButton


class GameLetters(View):
    
    def __init__(self, letters='greatly', min_len=5, max_len=7):
        sw,sh = State.screen.size
        x,y,w,h = settings.letters_dimensions
        View.__init__(self, subsurface_rect=Rect(x*sw, y*sh, w*sw, h*sh))
        
        # String of letters and corresponding buttons.
        self.value = ''
        self.letter_buttons = []
        self.free = []
        self.claimed = []
        self.set_value(letters)
        
        self.action_buttons = [
            ActionButton('Clear', self.clear_action, self),
            ActionButton('Shuffle', self.shuffle_action, self),
            ActionButton('Enter', self.enter_action, self),
        ]
        b = self.letter_buttons[-1]
        x = self.get_free_pos(b)[0] + b.rect.w
        y = 7
        for button in self.action_buttons:
            button.button_rect.topleft = x,y
            y += button.button_rect.height + 7
    
    def enter_action(self):
        answer = ''.join((b.char for b in self.claimed))
        if answer in State.flipcards:
            State.flipcards[answer].render()
            self.clear_action()
    
    def clear_action(self):
        idxes = [b.idx for b in self.claimed]
        idxes.sort()
        for i,b in enumerate(self.claimed):
            b.idx = idxes[i]
            pos = self.get_free_pos(b)
            b.free(pos)
            self.free[b.idx] = b
        del self.claimed[:]
    
    def shuffle_action(self):
        # Shuffle character order in the string.
        letters = [c for c in self.value]
        random.shuffle(letters)
        self.value = ''.join(letters)
        
        # Reorder button list to that of the string. Also, set button.idx value.
        new_buttons = []
        for i,c in enumerate(self.value):
            button = [b for b in self.letter_buttons if b.char == c][0]
            button.idx = i
            new_buttons.append(button)
            self.letter_buttons.remove(button)
        self.letter_buttons[:] = new_buttons
        
        # Recalculate button on-screen position if button is in free list.
        cx = self.rect.centerx
        cy = self.rect.bottom * 12/15
        length = len(self.value)
        new_buttons = []
        for button in self.letter_buttons:
            if button in self.free:
                pos = self.get_free_pos(button)
                button.shuffle(pos)
                new_buttons.append(button)
            else:
                new_buttons.append(None)
        self.free[:] = new_buttons
    
    def set_value(self, letters):
        self.value = letters
        
        del self.letter_buttons[:]
        font_size = 22
        font_name = data.filepath('font',settings.letters_font_name)
        font = pygame.font.Font(font_name, font_size)
        for idx,c in enumerate(letters):
            new_button = LetterButton(
                idx, c, font.render(c, True, Color('black')), self)
            self.letter_buttons.append(new_button)
            new_pos = self.get_free_pos(self.letter_buttons[0])
            new_button.position = new_button.move_to = new_pos
            self.free.append(new_button)
        
        self.shuffle_action()
    
    def get_free_pos(self, button):
        """Return rect.center coords."""
        idx = button.idx
        cx = self.rect.centerx
        cy = self.rect.bottom * 11/15
        length = len(self.letter_buttons)
        rect = button.rect
        width = rect.w + 5
        x = cx - width*length/2 + width*idx + width/2
        return x,cy
    
    def get_claimed_pos(self, button):
        """Return rect.center coords."""
        idx = self.claimed.index(button)
        cx = self.rect.centerx
        cy = self.rect.bottom * 4/15
        length = len(self.letter_buttons)
        rect = button.rect
        width = rect.w + 5
        x = cx - width*length/2 + width*idx + width/2
        return x,cy
    
    def pick_letter(self, char):
        for b in self.free:
            if b and b.char == char:
                self.pick(b)
                break
    
    def pick(self, button):
        self.free[button.idx] = None
        self.claimed.append(button)
        new_pos = self.get_claimed_pos(button)
        button.claim(new_pos)
    
    def pop(self, button):
        while 1:
            popped = self.claimed.pop()
            idx = popped.idx
            new_idx = [i for i in range(len(self.free)) if self.free[i] is None][0]
            popped.idx = new_idx
            self.free[new_idx] = popped
            new_pos = self.get_free_pos(popped)
            popped.free(new_pos)
            for swap in self.claimed:
                if swap.idx == new_idx: swap.idx = idx
            if popped is button:
                break
    
    def mouse_clicked(self, pos, button):
        for b in self.letter_buttons:
            if b.click(pos):
                if b in self.free:
                    self.pick(b)
                elif b in self.claimed:
                    self.pop(b)
                break
        for b in self.action_buttons:
            b.click(pos)
    
    def update(self, dt):
        for button in self.letter_buttons:
            button.think(dt)
    
    def draw(self, dt):
        if settings.debug_subsurfs:
            pygame.draw.rect(self.surface, Color('yellow'), self.rect, 1)
        for i in range(len(self.value)):
            button = self.letter_buttons[i]
            button.draw()
        for button in self.action_buttons:
            button.draw()
