import random

import pygame
from pygame.locals import *

from gummworld2 import State, Engine, View, Vec2d
from gummworld2 import context, data, geometry

import settings, gametitle, gameprogress, gameletters
import dictionary, pystardict, htmlscraper, textreader
from actionbutton import ActionButton
from backgrounddecorator import BackgroundDecorator
from statusmessage import StatusMessage
from letterbutton import FreeState, ClaimedState
from settings import keys


class Game(Engine):
    
    def __init__(self):
        Engine.__init__(self,
            frame_speed=settings.fps,
        )
        screen_rect = State.screen.rect
        
        # Default for Replay button, until something else is clicked.
        self.min_len = 3
        self.max_len = 4
        
        if settings.online_dictionary:
            if __debug__: print 'Game: using online dictionary'
            self.stardict = None
            self.html_parser = htmlscraper.SFOELD()
        elif settings.stardict_name:
            # Make pystardict to look up word definitions.
            if __debug__: print 'Game: using local StarDict'
            self.stardict = pystardict.Dictionary(
                settings.stardicts[settings.stardict_name])
            self.html_parser = None
        else:
            if __debug__: print 'Game: no dictionary configured'
            self.stardict = None
            self.html_parser = None
        self.pager = None
        self.pager_rect = Rect(0,0,600,400)
        self.pager_rect.center = screen_rect.center
        self.scroll_pager = 0
        
        # Pretty background.
        self.bg_decorator = BackgroundDecorator()
        
        # Status bar.
        self.message = StatusMessage(State.screen)
        
        # Start with only one view. Others are made when player starts a game.
        self.views = [
            gametitle.GameTitle(),
        ]
        self.game_title,self.game_progress,self.game_letters = self.views[0],None,None
        
        # Globally accessible flipcards.
        State.flipcards = {}
        
        # Draw the background in prep for command-line args, which start loading
        # word lists and making words immediately. We do not want the screen to
        # appear partially rendered when this happens.
        self.action_buttons = []
        for i,r in enumerate(self.bg_decorator.rects):
            self.bg_decorator.render(i)
        self.draw(0)
        
        # Action buttons...
        self.action_buttons = [
            None,
            ActionButton('Play 3-4', self.new_game_34, State.screen),
            ActionButton('Play 4-5', self.new_game_45, State.screen),
            ActionButton('Play 5-6', self.new_game_56, State.screen),
            ActionButton('Play 6-7', self.new_game_67, State.screen),
            ActionButton('Play 3-5', self.new_game_35, State.screen),
            ActionButton('Play 4-6', self.new_game_46, State.screen),
            ActionButton('Play 5-7', self.new_game_57, State.screen),
            ActionButton('Play 3-6', self.new_game_36, State.screen),
            ActionButton('Play 3-7', self.new_game_37, State.screen),
            ActionButton('Solve', self.solve, State.screen),
            ActionButton('Replay', self.replay, State.screen),
        ]
        
        # Make the word list action button...
        if settings.startword:
            self.picks = range(4,max(8,len(settings.startword)+1))
        else:
            self.picks = range(4,8)
        self.wordlist = settings.wordlist
        self.wordlist_name = ''
        self.next_wordlist()
        # Position the action buttons...
        button0 = self.action_buttons[0]
        button_height = button0.button_rect.h + 10
        height = len(self.action_buttons) * button_height
        button_rect = button0.button_rect
        o = screen_rect.left,screen_rect.centery
        top = Vec2d(screen_rect.right, screen_rect.centery-height/2-button_height)
        angle = geometry.angle_of(o, top)
        top += button_height
        angle1 = geometry.angle_of(o, top)
        arc_len = angle1 - angle
        r = screen_rect.w - button0.button_rect.w - 5
        a = angle
        for i,button in enumerate(self.action_buttons):
            button.button_rect.bottomleft = geometry.point_on_circumference(o, r, a)
            a += arc_len
        
        # Display credits every three minutes.
        State.clock.schedule_interval(self.show_credits, settings.credits_frequency)
        self.mouse_pos = 0,0
        
        if settings.startword:
            self.new_game(self.min_len, self.max_len, settings.startword)
    
    ## Button actions.
    def new_game_34(self): self.new_game(3,4)
    def new_game_45(self): self.new_game(4,5)
    def new_game_56(self): self.new_game(5,6)
    def new_game_67(self): self.new_game(6,7)
    def new_game_35(self): self.new_game(3,5)
    def new_game_46(self): self.new_game(4,6)
    def new_game_57(self): self.new_game(5,7)
    def new_game_36(self): self.new_game(3,6)
    def new_game_37(self): self.new_game(3,7)
    def replay(self):
        self.new_game(self.min_len, self.max_len, seed_len=len(settings.startword))
    
    def new_game(self, min_len, max_len, word=None, seed_len=0):
        self.min_len = min_len
        self.max_len = max_len
        
        if settings.fudge_max:
            # settings.fudge_max overrides other game logic
            self.min_len = 5
            self.max_len = 7
            if settings.wordlist == 'crossword.txt': letters = 'etsrido' # 66
            #if settings.wordlist == 'crossword.txt': letters = 'ednlasi' # 69
            #if settings.wordlist == 'crossword.txt': letters = 'tnashes' # 42
            #if settings.wordlist == 'crossword.txt': letters = 'lcsunae' # 40
            if settings.wordlist == 'single.txt': letters = 'paeitsr' # 171
        elif word:
            # __init__ calls with word=settings.startword
            letters = word
            self.min_len = settings.min_len
            self.max_len = settings.max_len
        elif seed_len:
            # replay() calls with seed_len=len(settings.startword)
            letters = self.words.random_pick(seed_len)
        else:
            # Player clicked a Play button
            settings.startword = ''
            letters = self.words.random_pick(self.max_len)
        
        message = 'Getting word list %d-%d' % (self.min_len,self.max_len)
        self.message.new_message(message, preempt=True, duration=2.)
        
        letters,combos = dictionary.random_words(
            letters, self.min_len, self.max_len, self.words)
        
        State.combos = combos
        
        # GameProgress needs to know its right edge so it doesn't place
        # flipcards over action buttons.
        button_rect = self.action_buttons[0].button_rect
        self.views[:] = [
            gametitle.GameTitle(),
            gameprogress.GameProgress(combos, button_rect.x),
            gameletters.GameLetters(letters),
        ]
        self.game_title,self.game_progress,self.game_letters = self.views
        if not settings.background:
            self.bg_decorator.update_all()
    
    def next_wordlist(self):
        if isinstance(self.wordlist, str):
            # Set from sys.argv in settings.py
            self.wordlist = settings.wordlists.index(self.wordlist)
            save_pos = 0,0
        elif self.wordlist_name == '':
            # Default from __init__
            self.wordlist = 0
            save_pos = 0,0
        else:
            self.wordlist += 1
            if self.wordlist >= len(settings.wordlists): self.wordlist = 0
            save_pos = self.action_buttons[0].button_rect.center
        filename = settings.wordlists[self.wordlist]
        self.wordlist_name = filename.split('.')[0].capitalize()
        new_button = ActionButton(
            self.wordlist_name, self.next_wordlist, State.screen)
        
        self.message.new_message('Loading word list '+self.wordlist_name,
            preempt=True, duration=2.)
        self.words = dictionary.Words(
            data.filepath('dictionary', filename), picks=self.picks)
        new_button.button_rect.center = save_pos
        self.action_buttons[0] = new_button
        self.message.new_message('%d words loaded'%self.words.count,
            preempt=True)
    
    def solve(self):
        for card in State.flipcards.values():
            card.render(color=Color(1,1,1))
    
    def update(self, dt):
        self.update_bg(dt)
        if self.pager:
            if self.scroll_pager:
                self.pager.scroll(self.scroll_pager)
            self.pager.update(dt)
        else:
            for view in self.views:
                view.update(dt)
        self.message.update(dt)
    
    def update_bg(self, dt):
        if settings.background:
            do_bg = True
            if self.game_letters:
                for b in self.game_letters.letter_buttons:
                    if b.current_state not in (FreeState,ClaimedState):
                        do_bg = False
            if do_bg:
                self.bg_decorator.update(dt)
    
    def draw(self, dt):
        State.screen.clear()
        for view in self.views:
            view.draw(dt)
        for button in self.action_buttons:
            button.draw()
        self.message.draw()
        if self.pager:
            self.pager.draw()
        State.screen.flip()
    
    def on_key_down(self, unicode, key, mod):
        if self.pager:
            if key == K_UP: self.scroll_pager = 1
            elif key == K_DOWN: self.scroll_pager = -1
            elif key == K_ESCAPE: self.pager = None
        elif key <= 255 and chr(key) in keys.letters:
            if self.game_letters:
                ## Note: this assume letters are in pygame contiguous order.
                i = key - K_a
                char = keys.letters[i]
                self.game_letters.pick_letter(char)
        elif key in keys.enter:
            if self.game_letters: self.game_letters.enter_action()
        elif key in keys.undo:
            if self.game_letters and self.game_letters.claimed:
                self.game_letters.pop(self.game_letters.claimed[-1])
        elif key in keys.clear:
            if mod & KMOD_SHIFT: self.on_quit()
            if self.game_letters: self.game_letters.clear_action()
        elif key in keys.shuffle:
            if self.game_letters: self.game_letters.shuffle_action()
        elif key in keys.replay:
            #self.new_game(self.min_len, self.max_len)
            self.replay()
        elif key in keys.sameword:
            self.new_game(self.min_len, self.max_len, self.letters)
    
    def on_key_up(self, key, mod):
        if self.pager:
            if key in (K_UP,K_DOWN): self.scroll_pager = 0
    
    def on_mouse_button_down(self, pos, button):
        if self.pager:
            if button == 4: self.pager.scroll(-1)
            elif button == 5: self.pager.scroll(1)
            else: self.pager = None
        else:
            if button > 3: return
            for view in self.views:
                if hasattr(view, 'mouse_clicked'):
                    view.mouse_clicked(pos, button)
            for b in self.action_buttons:
                b.click(pos)
            for card in State.flipcards.values():
                if card.click(pos):
                    if card.flipped:
                        # Look up definition.
                        self.message.new_message(
                            'Looking up word', duration=2, preempt=True,
                        )
                        key = ''.join(card.text)
                        self.get_definition(key)
                    elif button == 3:
                        # Flip the card.
                        card.flip()
                    else:
                        # Peek at next letter.
                        card.peek()
    
    def on_quit(self):
        context.pop()
    
    def get_definition(self, word):
        if self.stardict:
            if word in self.stardict:
                text = self.stardict[word.capitalize()]
                self.pager = textreader.TextReader(
                    text, self.pager_rect
                )
            else:
                self.message.new_message(
                    'Definition not found', duration=5, preempt=True,
                )
        elif self.html_parser:
            self.html_parser.lookup(word)
            text = []
            for t in self.html_parser.data:
                text.append(t)
                text.append('')
            self.pager = textreader.TextReader(
                text, self.pager_rect
            )
        else:
            self.message.new_message(
                'Dictionary not configured', duration=5, preempt=True,
            )
    
    def show_credits(self, dt):
        if not hasattr(self, 'credits_text'):
            fh = open(data.filepath('text','credits.txt'))
            self.credits_text = [s.rstrip() for s in fh.readlines()]
            self.credits_idx = 0
        text = self.credits_text[self.credits_idx]
        self.message.new_message(text, duration=settings.credits_duration)
        self.credits_idx += 1
        if self.credits_idx >= len(self.credits_text):
            self.credits_idx = 0
            if settings.credits_frequency < 90:
                settings.credits_frequency = 90
                State.clock.unschedule(self.show_credits)
                State.clock.schedule_interval(
                    self.show_credits, settings.credits_frequency)


def set_caption(dt):
    pygame.display.set_caption('%d fps' % State.clock.fps)
