#--------------------------------------------------------------
#
#   SimChip2 - Undo management
#
#--------------------------------------------------------------

from weakref import WeakKeyDictionary

undo_cache = WeakKeyDictionary()

def get_undo_manager(key):
	result = undo_cache.get(key)
	if not result:
		result = UndoManager()
		undo_cache[key] = result
	return result

def can_undo(chip):
	mgr = undo_cache.get(chip)
	return mgr is not None and mgr.can_undo()

def can_redo(chip):
	mgr = undo_cache.get(chip)
	return mgr is not None and mgr.can_redo()

def undo(chip):
	mgr = get_undo_manager(chip)
	mgr.undo(chip)

def redo(chip):
	mgr = get_undo_manager(chip)
	mgr.redo(chip)

#--------------------------------------------------------------

class UndoSelection(object):

	float_position = None

	def __init__(self, chip):
		self.selected_rect = chip.selected_rect
		flt = chip.floating_selection
		self.floating_selection = flt
		if flt:
			self.float_position = flt.position
	
	def restore(self, chip):
		contra = UndoSelection(chip)
		chip.invalidate_selection()
		chip.selected_rect = self.selected_rect
		flt = self.floating_selection
		chip.floating_selection = flt
		if flt:
			flt.position = self.float_position
		chip.changed()
		chip.invalidate_selection()
		return contra

#--------------------------------------------------------------

class UndoRect(UndoSelection):

	def __init__(self, chip, rect):
		UndoSelection.__init__(self, chip)
		self.rect = rect
		self.cells = chip.structure.get_rect(rect)
	
	def restore(self, chip):
		contra = UndoRect(chip, self.rect)
		UndoSelection.restore(self, chip)
		chip.set_rect(self.rect, self.cells)
		return contra

#--------------------------------------------------------------

class UndoCells(UndoSelection):

	def __init__(self, chip):
		UndoSelection.__init__(self, chip)
		self.map = {}
	
	def preserve_cell(self, chip, coords):
		map = self.map
		if coords not in map:
			map[coords] = chip.structure.get_cell(coords)
	
	def restore(self, chip):
		contra = UndoCells(chip)
		UndoSelection.restore(self, chip)
		cells = chip.structure.cells
		for coords, cell in self.map.iteritems():
			contra.preserve_cell(chip, coords)
			chip.set_cell(coords, cell)
		return contra

#--------------------------------------------------------------

class UndoManager(object):

	def __init__(self):
		self.undo_stack = []
		self.redo_stack = []
		self.open_cells = None
	
	def preserve_rect(self, chip, rect):
		self.push_undo(UndoRect(chip, rect))

	def preserve_cell(self, chip, coords):
		item = self.open_cells
		if not item:
			item = UndoCells(chip)
			self.push_undo(item)
			self.open_cells = item
		item.preserve_cell(chip, coords)
	
	def preserve_selection(self, chip):
		self.push_undo(UndoSelection(chip))
	
	def push_undo(self, item):
		self.undo_stack.append(item)
		del self.redo_stack[:]
		self.open_cells = None		

	def finish_editing(self):
		self.open_cells = None

	def can_undo(self):
		return bool(self.undo_stack)

	def can_redo(self):
		return bool(self.redo_stack)

	def undo(self, chip):
		self.transfer(chip, self.undo_stack, self.redo_stack)
	
	def redo(self, chip):
		self.transfer(chip, self.redo_stack, self.undo_stack)
	
	def transfer(self, chip, from_stack, to_stack):
		if from_stack:
			item = from_stack.pop()
			contra = item.restore(chip)
			to_stack.append(contra)
		self.open_cells = None
