#--------------------------------------------------------------
#
#   SimChip2 - Testing
#
#--------------------------------------------------------------

from __future__ import division
from numpy import zeros, int32
from GUI import Window, View, Label, Button, Row, Column, Grid, Font
from GUI.Geometry import offset_rect
from resources import get_image
from simulation import Vcc
import www

Vt = 0.5 * Vcc

#--------------------------------------------------------------

class TestResultsView(View):

	background = get_image("test_results.png")
	margin = 15
	font = Font("Courier New", 14)
	stamps = [
		get_image("failed-stamp.png"),
		get_image("passed-stamp.png")
	]

	def __init__(self, game):
		self.game = game
		m = 2 * self.margin
		w, h = self.background.size
		View.__init__(self, width = w + m, height = h + m)
	
	def draw(self, canv, ur):
		canv.erase_rect(ur)
		bg = self.background
		src = bg.bounds
		m = self.margin
		dst = offset_rect(src, (m, m))
		bg.draw(canv, src, dst)
		level = self.game.current_level
		self.draw_text(canv, 96, 96, 'l', "%s", level.title)
		results = level.test_results
		if results:
			self.draw_text(canv, 208, 144, 'r', "%d", results.correctness)
			self.draw_text(canv, 480, 168, 'r', "%.3f", results.supply_current)
			self.draw_text(canv, 480, 208, 'r', "%.1f", results.propagation_delay)
			self.draw_text(canv, 480, 248, 'r', "%d", results.silicon_area)
			stamp = self.stamps[results.passed()]
			src = stamp.bounds
			w, h = stamp.size
			x = 144 - w // 2
			y = 240 - h // 2
			dst = (x, y, x + w, y + h)
			stamp.draw(canv, src, dst)
	
	def draw_text(self, canv, x, y, align, fmt, value):
		if value is not None:
			text = fmt % value
			f = self.font
			by = y + f.height / 2 - f.descent
			if align == 'r':
				bx = x - f.width(text)
			else:
				bx = x
			canv.font = f
			canv.moveto(bx, by)
			canv.show_text(text)

#--------------------------------------------------------------

class TestResultsWindow(Window):

	keeps_document_open = False

	game = property(lambda self: self.document)

	def __init__(self, game):
		Window.__init__(self, document = game)
		self.result_view = TestResultsView(game)
		self.next_button = Button("Next Level", action = 'next_level')
		button_list = [None, self.next_button]
		if www.debug:
			button_list.append(Button("Submit", action = 'submit'))
		buttons = Row(button_list, expand = 0, padding = (10, 10))
		contents = Column([self.result_view, buttons], spacing = 0, align = 'r')
		self.add(contents)
		self.shrink_wrap()
		game.add_view(self)
		self.current_level_changed()
	
	def close_cmd(self):
		self.hide()
	
	def current_level_changed(self, *args):
		self.update_title()
		self.test_results_changed()
	
	def test_results_changed(self, *args):
		self.result_view.invalidate()
		self.update_buttons()
	
	def update_title(self):
		level = self.game.current_level
		self.title = "Test Results - Level %s" % (level.index + 1)
	
	def update_buttons(self):
		self.next_button.enabled = self.game.next_level_is_unlocked()
	
	def next_level(self):
		self.game.select_next_level()
		self.close_cmd()
	
	def submit(self):
		chip = self.game.current_chip
		level = self.game.current_level
		www.submit_results(chip, level.test_results)

#--------------------------------------------------------------

class TestResults(object):
	#  correctness         int     percentage
	#  propagation_delay   float   ns
	#  silicon_area        int     sq um
	
	correctness = 0
	propagation_delay = 0
	silicon_area = 0
	
	def passed(self):
		return self.correctness >= 100

#--------------------------------------------------------------

def percent(good, bad):
	return (100 * good) // (good + bad)

def propdelay(ntrans, res, good, bad):
	return res * bad / ntrans / 10

def check_test_results(level):
	chip = level.chip
	setup = level.test_setup
	waveforms = setup.waveforms
	resolution = waveforms.resolution
	num_samples = waveforms.num_samples
	time = resolution * num_samples
	count = zeros([2, 2], int32)
	ntrans = 0
	for exp in setup.expectations.tracks:
		p = exp.pin_index
		out = waveforms.track_for_pin(p)
		s_exp = exp.samples
		s_out = out.samples
		n_exp = min(len(s_exp), exp.end)
		n_out = min(len(s_out), out.end)
		ntrans += sum((s_exp[1:] ^ s_exp[:-1]) & 1)
		for i in xrange(min(n_exp, n_out)):
			e = s_exp[i]
			if not (e & 4):
				j = (e & 2) >> 1
				k = (s_out[i] > Vt) <> (e & 1)
				#print "Pin %2d %4d: e = %d  v = %d    j = %d k = %d" % (p, i, e, s_out[i] > Vt, j, k) ###
				count[j, k] += 1
	vcc_pin = level.chip_params.vcc_pin
	print "check_test_results:" ###
	print "count =", count ###
	print "num transitions =", ntrans ###
	print "vcc_pin =", vcc_pin ###
	results = TestResults()
	results.correctness = percent(*count[0])
	results.supply_current = chip.measure_supply_current(vcc_pin, time)
	results.propagation_delay = propdelay(ntrans, resolution, *count[1])
	results.silicon_area = level.chip.measure_silicon_area()
	print "Correctness: %3d%%" % results.correctness ###
	print "Supply current: %.3f" % (results.supply_current or 0) ###
	print "Propagation delay: %.1f" % results.propagation_delay ###
	print "Silicon area: %s" % results.silicon_area ###
	level.set_test_results(results)
	return results
