#--------------------------------------------------------------
#
#   SimChip2 - Game
#
#--------------------------------------------------------------

import os, sys
import cPickle as pickle
from GUI import Document, Model
from GUI.Files import DirRef
from GUI.Resources import find_resource
from chip import Chip
from level import Level
from level_list_view import choose_level
from doc_browser_window import DocBrowserWindow
from waveform_window import WaveformWindow
from testing import TestResultsWindow

def destroy(obj):
	if obj:
		obj.destroy()

def find_default_game_save_dir():
	d = os.path.dirname
	path1 = d(d(os.path.abspath(sys.path[0])))
	path2 = os.path.join(path1, "Saves")
	if os.path.isdir(path2):
		return path2
	else:
		return os.path.expanduser("~")

default_game_save_dir = DirRef(find_default_game_save_dir())
#print "game: default_game_save_dir =", default_game_save_dir ###

#--------------------------------------------------------------

class GameState(Model):
	#  chip_map            {key: Chip}
	#  current_level_key   string
	#  level_page_indexes  {key: int}
	#  completed_levels    {key}
	#  unlocked_levels     {key}
	#  test_results        {key: TestResults}
	
	level_page_indexes = {} # Fallback for old unpickled states
		
	current_chip = property(lambda self:
		self.get_chip_for_key(self.current_level_key))
	
	game = property(lambda self: self.parent)
	
	def __init__(self, game):
		Model.__init__(self, game)
		self.chip_map = {}
		self.current_level_key = game.level_list[0].key
		self.level_page_indexes = {}
		self.completed_levels = set()
		#self.unlocked_levels = set()
		self.test_results = {}
		#self.init_level_sets()
	
	def __setstate__(self, d):
		self.__dict__.update(d)
		for key, chip in self.chip_map.iteritems():
			chip.level_key = key
		if 'completed_levels' not in d:
			self.completed_levels = set()
			#self.unlocked_levels = set()
		if 'test_results' not in d:
			self.test_results = {}
	
	def post_unpickle(self):
		pass
		#if not self.unlocked_levels:
		#	self.init_level_sets()

#	def init_level_sets(self):
#		for level in self.game.level_list:
#			if level.initially_unlocked:
#				self.unlocked_levels.add(level.key)
	
	def get_chip_for_key(self, key):
		chip = self.chip_map.get(key)
		if not chip:
			params = self.parent.get_chip_params_for_level(key)
			chip = Chip(self, key, params)
			self.chip_map[key] = chip
		return chip
	
	def test_results_for_level(self, level):
		return self.test_results.get(level.key)

	def set_test_results_for_level(self, level, results):
		self.test_results[level.key] = results

	def notify(self, *args):
		self.parent.notify(*args)

#--------------------------------------------------------------

class GameDoc(Document):
	#  state        GameState
	#  level_list   [Level]
	#  level_map    {key: Level}

	app_magic = "SmCh"
	doc_magic = "GAME"
	doc_version = 1

	doc_browser_window = None
	waveform_window = None
	test_results_window = None
	visit_any_level = False

	current_chip = property(lambda self: self.state.current_chip)
	current_setup = property(lambda self: self.current_chip.current_setup)
	current_waveforms = property(lambda self: self.current_chip.current_waveforms)
	current_level_html = property(lambda self: self.get_current_level_html())

	current_level = property(lambda self:
		self.level_map.get(self.state.current_level_key))
	
	current_level_page_index = property(
		lambda self: self.get_current_level_page_index(),
		lambda self, i: self.set_current_level_page_index(i)
		)
		
	current_level_num_pages = property(lambda self:
		self.get_current_level_num_pages())
	
	game_speed = property(
		lambda self: self.current_chip.game_speed,
		lambda self, x: self.set_game_speed(x))

	def __init__(self):
		Document.__init__(self)
		self.load_levels()
	
	def destroy(self):
		destroy(self.doc_browser_window)
		destroy(self.waveform_window)
		destroy(self.test_results_window)
		Document.destroy(self)
	
	def new_contents(self):
		self.state = GameState(self)
	
	def destroy_contents(self):
		self.state.destroy()
	
	def notify(self, *args):
		#print "GameDoc.notify:", args ###
		self.notify_views(*args)

	def write_contents(self, f):
		f.write(self.app_magic)
		f.write(self.doc_magic)
		f.write("%04d" % self.doc_version)
		pickle.dump(self.state, f)

	def read_contents(self, f):
		app_magic = f.read(4)
		doc_magic = f.read(4)
		file_version = int(f.read(4))
		state = pickle.load(f)
		state.parent = self
		self.state = state
		state.post_unpickle()
		#<<<
		chip = self.chip_for_level(self.level_list[1])
		area = chip.measure_silicon_area()
		print "Game: silicon area =", area
		#>>>
	
	def make_title(self):
		title = Document.make_title(self).replace("Untitled", "SimChipGame")
		if title.endswith("-1"):
			title = title[:-2]
		return title
	
	def get_default_save_filename(self):
		return self.title
	
	def get_default_save_directory(self):
		return default_game_save_dir
	
	def load_levels(self):
		level_dir = find_resource("levels")
		filenames = os.listdir(level_dir)
		levels = []
		map = {}
		i = 0
		for name in filenames:
			path = os.path.join(level_dir, name)
			if os.path.isdir(path):
				level = Level(self, i, path)
				levels.append(level)
				map[level.key] = level
				i += 1
		self.level_list = levels
		self.level_map = map
	
	def get_level_with_index(self, i):
		levels = self.level_list
		if 0 <= i < len(levels):
			return levels[i]
	
	def select_level(self, level, force = False):
		if level:
			#if force or self.visit_any_level:
			#	level.mark_unlocked()
			#if level.is_unlocked():
			if force or self.visit_any_level or level.is_unlocked():
				self.state.current_level_key = level.key
				self.hide_waveforms()
				self.hide_test_results()
				self.show_instructions()
				self.notify('current_level_changed')
				return True
		return False
	
	def select_level_number(self, i, force = False):
		levels = self.level_list
		if 1 <= i <= len(levels):
			self.select_level(levels[i - 1], force)
	
	def select_next_level(self):
		self.select_level(self.get_next_level())
	
	def set_game_speed(self, x):
		self.current_chip.game_speed = x
		self.changed()
		self.notify('game_speed_changed')

	def begin_frame(self):
		chip = self.current_chip
		if chip:
			chip.begin_frame()
	
	def level_for_key(self, key):
		return self.level_map.get(key)
	
	def chip_for_level(self, level):
		return self.state.get_chip_for_key(level.key)
	
	def get_chip_params_for_level(self, key):
		level = self.level_for_key(key)
		if level:
			return level.chip_params

	def get_current_level_html(self):
		i = self.current_level_page_index
		html = self.current_level.html
		if 0 <= i < len(html):
			return html[i]
	
	def get_current_level_page_index(self):
		return self.state.level_page_indexes.get(self.current_level.key) or 0
	
	def set_current_level_page_index(self, i):
		i = max(0, min(i, self.current_level_num_pages - 1))
		self.state.level_page_indexes[self.current_level.key] = i
		self.changed()
		self.notify('current_level_page_changed')
	
	def get_current_level_num_pages(self):
		return len(self.current_level.html)

	def setup_menus(self, m):
		Document.setup_menus(self, m)
		m.choose_level_cmd.enabled = True
		m.instructions_cmd.enabled = True
		m.waveforms_cmd.enabled = True
		m.test_results_cmd.enabled = self.current_level.test_results is not None

	def choose_level_cmd(self):
		choose_level(self)
	
	def instructions_cmd(self):
		self.show_instructions()
	
	def waveforms_cmd(self):
		self.show_waveforms()
	
	def test_results_cmd(self):
		self.show_test_results()

	def next_level_is_unlocked(self):
		level = self.get_next_level()
		return level is not None and level.is_unlocked()
	
#	def unlock_level_after(self, level):
#		next_level = self.level_after(level)
#		if next_level:
#			next_level.mark_unlocked()
	
	def get_next_level(self):
		return self.level_after(self.current_level)
	
	def level_after(self, level):
		return self.level_with_index(level.index + 1)
	
	def at_last_level(self):
		return self.get_next_level() is not None

	def level_with_index(self, i):
		levels = self.level_list
		if 0 <= i < len(levels):
			return levels[i]
	
	def hide_window(self, win):
		if win:
			win.hide()
	
	def show_workbench(self):
		self.windows[0].show()
	
	def show_instructions(self):
		browser = self.doc_browser_window
		if not browser:
			browser = DocBrowserWindow(self, auto_position = False)
			self.doc_browser_window = browser
			chip_win = self.windows[0]
			browser.position = (chip_win.left + 100, chip_win.top)
		browser.show()
	
	def hide_instructions(self):
		self.hide_window(self.doc_browser_window)
	
	def show_waveforms(self):
		win = self.waveform_window
		if not win:
			win = WaveformWindow(self)
			self.waveform_window = win
		win.show()
	
	def hide_waveforms(self):
		self.hide_window(self.waveform_window)

	def show_test_results(self):
		win = self.test_results_window
		if not win:
			win = TestResultsWindow(self)
			self.test_results_window = win
		win.show()
	
	def hide_test_results(self):
		self.hide_window(self.test_results_window)

	def run_test(self):
		chip = self.current_chip
		chip.select_test_setup()
		self.show_waveforms()
		chip.auto_stop()
		chip.reset_supply_metering()
		chip.auto_start()

	def test_results_for_level(self, level):
		return self.state.test_results_for_level(level)
	
	def set_test_results_for_level(self, level, results):
		self.state.set_test_results_for_level(level, results)
		self.changed()
