#--------------------------------------------------------------
#
#   SimChip2 - Chip editing and simulation view
#
#--------------------------------------------------------------

from GUI import ScrollableView, rgb, Font, StdColors, application
from GUI.Geometry import offset_rect, empty_rect, sect_rect, pt_in_rect, sub_pt
#from resources import get_image
from simulation import _MET_, _SIL_, MET, SIL, VIA, MS, HC, VC, Snippet
from rendering import get_render_buffer, SnippetRenderBuffer
from undo import can_undo, can_redo, undo, redo
import tools

#def get_sym(name):
#	return get_image("sym-" + name + ".png") 
#
#sym_via = get_sym("via")
#sym_metal = get_sym("metal")
#sym_silicon = [get_sym(n) for n in ("p", "p+", "n+", "n")]

tool_material_ops = {
	tools.METAL: (MET, 1),
	tools.SIL_N: (SIL, -1),
	tools.SIL_NPLUS: (SIL, -2),
	tools.SIL_P: (SIL, +1),
	tools.SIL_PPLUS: (SIL, +2),
	tools.VIA: (VIA, 1),
}

tool_connection_ops = {
	tools.METAL: MET,
	tools.SIL_N: SIL,
	tools.SIL_NPLUS: SIL,
	tools.SIL_P: SIL,
	tools.SIL_PPLUS: SIL,
}

tool_for_key = {
	"m": tools.METAL,
	"M": tools.METAL,
	"p": tools.SIL_P,
	"P": tools.SIL_PPLUS,
	"n": tools.SIL_N,
	"N": tools.SIL_NPLUS,
	"v": tools.VIA,
	"V": tools.VIA,
}

class ChipView(ScrollableView):

	cell_size = 16
	substrate_color = rgb(0.4, 0.4, 0.4)
	grid_color = rgb(0.5, 0.5, 0.5)
	sel_color = StdColors.yellow
	pad_font = Font("Sans", 9)
	
	chip = property(lambda self: self.window.chip)
	current_tool = property(lambda self: self.window.selected_tool)
	selected_rect = property(lambda self: self.chip.selected_rect)
	floating_selection = property(lambda self: self.chip.floating_selection)
	
	first_coords = None
	last_coords = None
	#selected_rect = None
	#floating_selection = None
	drag_offset = None
	
	def __init__(self, **kwds):
		ScrollableView.__init__(self, border = True, **kwds)

	def update_extent(self):
		s = self.cell_size
		chip = self.chip
		nrows, ncols = chip.size
		#print "ChipView.update_extent:", nrows, "rows by", ncols, "cols, cell size =", s ###
		self.extent = (s * ncols, s * nrows)
		self.invalidate()
	
	def tool_changed(self):
		self.drop_selection()
		tool = self.window.selected_tool
		if tool == tools.CONFIGURE:
			cursor = None
		else:
			cursor = tools.cursors.get(tool)
		self.cursor = cursor
	
	def setup_menus(self, m):
		ScrollableView.setup_menus(self, m)
		sel = self.any_selection()
		data = application().get_clipboard()
		valid = Snippet.is_valid_string(data)
		chip = self.chip
		m.cut_cmd.enabled = sel
		m.copy_cmd.enabled = sel
		m.paste_cmd.enabled = valid
		m.clear_cmd.enabled = sel
		m.flip_horizontal_cmd.enabled = sel
		m.flip_vertical_cmd.enabled = sel
		m.rotate_left_cmd.enabled = sel
		m.rotate_right_cmd.enabled = sel
		m.undo_cmd.enabled = can_undo(chip)
		m.redo_cmd.enabled = can_redo(chip)

	def cut_cmd(self):
		snip = self.chip.cut_snippet()
		self.put_snippet_on_clipboard(snip)
	
	def copy_cmd(self):
		snip = self.chip.copy_snippet()
		self.put_snippet_on_clipboard(snip)
	
	def paste_cmd(self):
		data = application().get_clipboard()
		snip = Snippet.from_string(data)
		if snip:
			sel = self.get_selection_rect()
			if sel:
				snip.center_in_rect(sel)
			else:
				coords = self.first_coords
				if coords:
					snip.center_on_coords(coords)
				else:
					self.center_snippet_in_view(snip)
			self.chip.paste_snippet(snip)
	
	def clear_cmd(self):
		self.chip.cut_snippet()
	
	def flip_horizontal_cmd(self):
		self.chip.flip_horizontal()
	
	def flip_vertical_cmd(self):
		self.chip.flip_vertical()
	
	def rotate_left_cmd(self):
		self.chip.rotate_left()
	
	def rotate_right_cmd(self):
		self.chip.rotate_right()
	
	def undo_cmd(self):
		undo(self.chip)
	
	def redo_cmd(self):
		redo(self.chip)
	
	def center_snippet_in_view(self, snip):
		vr = sect_rect(self.viewed_rect(), self.extent_rect())
		vc = self.view_to_cell_rect(vr)
		snip.center_in_rect(vc)
	
	def extent_rect(self):
		w, h = self.extent
		return (0, 0, w, h)
	
	def put_snippet_on_clipboard(self, snip):
		if snip:
			data = snip.to_string()
			application().set_clipboard(data)		
	
	def mouse_down(self, event):
		tool = self.effective_tool(event)
		coords = self.mouse_cell(event)
		if coords:
			self.first_coords = coords
			if tool == tools.SELECT:
				sel = self.get_selection_rect()
				if sel and pt_in_rect(coords, sel):
					self.drag_offset = sub_pt(coords, sel[:2])
					self.chip.begin_selection_drag(copy = event.option)
				else:
					self.drop_selection()
			else:
				self.drop_selection()
				op = tool_material_ops.get(tool)
				if op:
					self.add_or_remove_material(op, coords, event)
					self.last_coords = coords
	
	def mouse_drag(self, event):
		tool = self.effective_tool(event)
		if tool == tools.SELECT:
			this = self.view_to_cell_pt(event.position)
			if self.drag_offset:
				self.chip.position_floating_selection(sub_pt(this, self.drag_offset))
			else:
				first = self.first_coords
				if first and this:
					self.select_rect(self.make_selection_rect(first, this))
		else:
			op = tool_material_ops.get(tool)
			if op:
				this = self.mouse_cell(event, constrain = event.shift)
				if this:
					last = self.last_coords
					if last and last <> this:
						layer = tool_connection_ops.get(tool)
						if not event.option and layer is not None:
							#print "ChipView: making connections between", last, "and", this ###
							for coords, direction in self.connections_between(last, this):
								#print "...coords =", coords, "layer =", layer, "direction =", direction ###
								self.add_connection(coords, layer, direction)
						#print "ChipView: adding material between", last, "and", this ###
						for coords in self.coords_between(last, this):
							#print "...coords =", coords ###
							self.add_or_remove_material(op, coords, event)
					self.last_coords = this
	
	def mouse_up(self, event):
		self.drag_offset = None
		self.chip.finish_editing()

	def key_down(self, event):
		c = event.char
		if c == "\r":
			self.drop_selection()
		elif c == "\b" or c == "\x7f":
			self.clear_cmd()
		else:
			tool = tool_for_key.get(c)
			if tool:
				self.window.select_tool(tool)

	def mouse_cell(self, event, constrain = False):
		x, y = event.position
		cs = self.cell_size
		row = int(y // cs)
		col = int(x // cs)
		nrows, ncols = self.chip.size
		if 0 <= row < nrows and 0 <= col < ncols:
			if constrain:
				row0, col0 = self.first_coords
				if abs(row - row0) < abs(col - col0):
					row = row0
				else:
					col = col0
			return (row, col)
	
	def view_to_cell_pt(self, p):
		x, y = p
		cs = self.cell_size
		row = int(y // cs)
		col = int(x // cs)
		return (row, col)
	
	def cell_to_view_pt(self, cell):
		row, col = cell
		cs = self.cell_size
		return (col * cs, row * cs)
	
	def cell_to_view_rect(self, rect):
		l, t = self.cell_to_view_pt(rect[:2])
		r, b = self.cell_to_view_pt(rect[2:])
		return (l, t, r, b)
	
	def view_to_cell_rect(self, rect):
		t, l = self.view_to_cell_pt(rect[:2])
		b, r = self.view_to_cell_pt(rect[2:])
		return (t, l, b, r)
	
	def effective_tool(self, event):
		if event.control:
			return tools.VIA
		else:
			return self.current_tool
	
	def connections_between(self, coords1, coords2):
		row1, col1 = coords1
		row2, col2 = coords2
		if row1 == row2:
			direction = HC
			for col in xrange(min(col1, col2), max(col1, col2)):
				yield (row1, col), direction
		elif col1 == col2:
			direction = VC
			for row in xrange(min(row1, row2), max(row1, row2)):
				yield (row, col1), direction
	
	def coords_between(self, coords1, coords2):
		row1, col1 = coords1
		row2, col2 = coords2
		drow = dcol = 0
		if row1 == row2:
			dcol = 1 if col1 < col2 else -1
		elif col1 == col2:
			drow = 1 if row1 < row2 else -1
		if drow or dcol:
			while col1 <> col2 or row1 <> row2:
				row1 += drow
				col1 += dcol
				yield (row1, col1)
		else:
			yield (row2, col2)
	
	def add_or_remove_material(self, op, coords, event):
		layer, value = op
		if event.option:
			value = 0
		self.chip.add_or_remove_material(coords, layer, value)
	
	def add_connection(self, coords, layer, direction):
		self.chip.add_connection(coords, layer, direction)
	
	def invalidate_cell(self, coords):
		#print "ChipView.invalidate_cell:", coords ###
		row, col = coords
		cs = self.cell_size
		x = col * cs
		y = row * cs
		self.invalidate_rect((x, y, x + cs, y + cs))
	
	def any_selection(self):
		return self.chip.any_selection()

	def select_rect(self, sel):
		self.chip.select_rect(sel)

	def drop_selection(self):
		self.chip.drop_selection()
	
	def get_selection_rect(self):
		return self.chip.get_selection_rect()
	
	def invalidate_cells(self, r):
		self.invalidate_rect(self.cell_to_view_rect(r))

	def draw(self, canv, update_rect):
		self.draw_chip(canv, update_rect)
		self.draw_floating_selection(canv)
		sel = self.get_selection_rect()
		if sel:
			self.outline_selection(canv, sel)

	def draw_chip(self, canv, update_rect):
		canv.backcolor = self.substrate_color
		canv.erase_rect(update_rect)
		cs = self.cell_size
		h = cs // 2
		chip = self.chip
		nrows, ncols = chip.size
		maxrow = nrows - 1
		maxcol = ncols - 1
		col0 = max(0, int(update_rect[0] // cs))
		row0 = max(0, int(update_rect[1] // cs))
		col1 = min(ncols, int(update_rect[2] // cs + 1))
		row1 = min(nrows, int(update_rect[3] // cs + 1))
		row_range = xrange(row0, row1)
		col_range = xrange(col0, col1)
		canv.pencolor = self.grid_color
		canv.newpath()
		width, height = self.extent
		pad_width = 3 * cs
		for row in row_range:
			y = row * cs + 0.5
			canv.moveto(pad_width, y)
			canv.lineto(width - pad_width, y)
		for col in xrange(max(3, col0), min(ncols - 2, col1)):
			x = col * cs + 0.5
			canv.moveto(x, 0)
			canv.lineto(x, height)
		canv.stroke()
		buf = get_render_buffer(chip)
		if self.window.show_voltages:
			buf.v_silicon_image.draw(canv, update_rect, update_rect)
			buf.v_metal_image.draw(canv, update_rect, update_rect)
		else:
			buf.silicon_image.draw(canv, update_rect, update_rect)
			buf.metal_image.draw(canv, update_rect, update_rect)
		buf.via_image.draw(canv, update_rect, update_rect)
		font = self.pad_font
		canv.font = font
		yoff = font.ascent // 2
		for pin_no, (row, col) in enumerate(chip.pads):
			label = chip.name_of_pin(pin_no)
			x = col * cs + h - font.width(label) // 2
			y = row * cs + h + yoff
			canv.moveto(x, y)
			canv.show_text(label)

	def draw_floating_selection(self, canv):
		flt = self.floating_selection
		if flt:
			buf = flt.render_buffer
			if not buf:
				buf = SnippetRenderBuffer(flt)
				buf.update(flt)
				flt.render_buffer = buf
			src = buf.get_image_bounds()
			r = flt.bounds
			dst = self.cell_to_view_rect(r)
			buf.silicon_image.draw(canv, src, dst)
			buf.metal_image.draw(canv, src, dst)
			buf.via_image.draw(canv, src, dst)

	def outline_selection(self, canv, sel):
		canv.forecolor = self.sel_color
		canv.frame_rect(self.cell_to_view_rect(sel))

	def make_selection_rect(self, coords1, coords2):
		nr, nc = self.chip.size
		r1, c1 = coords1
		r2, c2 = coords2
		r = (max(0, min(r1, r2)), max(3, min(c1, c2)),
			min(nr, max(r1, r2) + 1), min(nc - 3, max(c1, c2) + 1))
		if not empty_rect(r):
			return r

	#--------------------------------------------------------------

#	def draw_direct(self, canv, update_rect):
#		#print "ChipView.draw:", update_rect ###
#		canv.backcolor = self.substrate_color
#		canv.erase_rect(update_rect)
#		cs = self.cell_size
#		h = cs // 2
#		via_src = sym_via.bounds
#		via_offx = h - via_src[2] // 2
#		via_offy = h - via_src[3] // 2
#		chip = self.chip
#		struct = chip.structure
#		cells = struct.cells
#		metal = cells[..., _MET_]
#		silicon = cells[..., _SIL_]
#		vias = cells[..., VIA]
#		nrows, ncols = chip.size
#		maxrow = nrows - 1
#		maxcol = ncols - 1
#		col0 = max(0, int(update_rect[0] // cs))
#		row0 = max(0, int(update_rect[1] // cs))
#		col1 = min(ncols, int(update_rect[2] // cs + 1))
#		row1 = min(nrows, int(update_rect[3] // cs + 1))
#		row_range = xrange(row0, row1)
#		col_range = xrange(col0, col1)
#		canv.pencolor = self.grid_color
#		canv.newpath()
#		width, height = self.extent
#		pad_width = 3 * cs
#		for row in row_range:
#			y = row * cs + 0.5
#			canv.moveto(pad_width, y)
#			canv.lineto(width - pad_width, y)
#		for col in xrange(max(3, col0), min(ncols - 2, col1)):
#			x = col * cs + 0.5
#			canv.moveto(x, 0)
#			canv.lineto(x, height)
#		canv.stroke()
#
#		def draw_cell(layer, sym):
#			left = col > 0 and layer[row, col-1, HC]
#			up = row > 0 and layer[row-1, col, VC]
#			right = col < maxcol and layer[row, col, HC]
#			down = row < maxrow and layer[row, col, VC]
#			upleft = up and left and layer[row-1, col-1, HC] and layer[row-1, col-1, VC]
#			upright = up and right and layer[row-1, col, HC] and layer[row-1, col+1, VC]
#			downleft = down and left and layer[row+1, col-1, HC] and layer[row, col-1, VC]
#			downright = down and right and layer[row+1, col, HC] and layer[row, col+1, VC]
#
#			def draw_quarter(i, j, hcon, vcon, hvcon):
#				r = i << 1 | j
#				c = 4 if hvcon else hcon | vcon << 1
#				sx = c * h
#				sy = r * h
#				src = (sx, sy, sx + h, sy + h)
#				dx = x + j * h
#				dy = y + i * h
#				dst = (dx, dy, dx + h, dy + h)
#				sym.draw(canv, src, dst)
#
#			# draw_cell:
#			draw_quarter(0, 0, left, up, upleft)
#			draw_quarter(0, 1, right, up, upright)
#			draw_quarter(1, 0, left, down, downleft)
#			draw_quarter(1, 1, right, down, downright)
#
#		def draw_via():
#			dst = offset_rect(via_src, (x + via_offx, y + via_offy))
#			sym_via.draw(canv, via_src, dst)
#
#		# draw:
#		for row in row_range:
#			y = row * cs
#			for col in col_range:
#				x = col * cs
#				s = silicon[row, col, MS]
#				if s:
#					draw_cell(silicon, sym_silicon[(s % 5) - 1])
#				if metal[row, col, MS]:
#					draw_cell(metal, sym_metal)
#				if vias[row, col]:
#					draw_via()
#		font = self.pad_font
#		canv.font = font
#		yoff = font.ascent // 2
#		for pin_no, (row, col) in enumerate(chip.pads):
#			label = chip.name_of_pin(pin_no)
#			x = col * cs + h - font.width(label) // 2
#			y = row * cs + h + yoff
#			canv.moveto(x, y)
#			canv.show_text(label)

#--------------------------------------------------------------

