#--------------------------------------------------------------
#
#   SimChip2 - Application
#
#--------------------------------------------------------------

from __future__ import division
import sys
from time import time, clock
from GUI import Application, Menu, Task
from GUI.Files import FileType
from GUI.StdMenus import basic_menus, file_cmds
from constants import frame_rate
from game import GameDoc, default_game_save_dir
from chip_window import ChipWindow
from doc_browser_window import DocBrowserWindow
#from level_list_view import LevelListWindow
from oobe import OOBDialog
from timing import cpu_time
import profile

profiling = 0
min_frame_rate = 1
max_frame_rate = 60

game_menu = Menu("Game", [
	("Choose Level/L", 'choose_level_cmd'),
	"-",
	("Show Instructions/I", 'instructions_cmd'),
	("Show Waveforms/G", 'waveforms_cmd'),
	("Show Test Results/R", 'test_results_cmd'),
])

std_menus = basic_menus(include = file_cmds,
	substitutions = {
		'new_cmd': "New Game",
		'open_cmd': "Restore Game...",
		'save_cmd': "Save Game",
		'save_as_cmd': "Save Game As...",
	}
)

edit_menu = std_menus.menu_with_command('copy_cmd')

edit_menu.extend([
	"-",
	("Flip Horizontal/^H", 'flip_horizontal_cmd'),
	("Flip Vertical/^V", 'flip_vertical_cmd'),
	("Rotate Left/^L", 'rotate_left_cmd'),
	("Rotate Right/^R", 'rotate_right_cmd'),
])

menus = std_menus + [game_menu]

game_file_type = FileType(
	name = "SimChip Saved Game",
	suffix = "simchip",
	mac_creator = "SChp",
	mac_type = "SChG")

#--------------------------------------------------------------

class SC2Application(Application):

	worst_percent_busy = 0
	last_frame_begin_time = 0
	last_cpu_time = 0
	frame_timer = None
	sample_delay = 0 #5

	def __init__(self):
		Application.__init__(self, title = "SimChip")
		self.open_file_types = [game_file_type]
		self.save_file_type = game_file_type
		self.menus = menus
		self.average_frame_cpu_time = 1.0 / frame_rate
		self.init_frame_timer()
		if "-n" in sys.argv:
			self.new_cmd()
	
	def init_frame_timer(self):
		if self.frame_timer:
			self.frame_timer.stop()
		frame_time = 1.0 / frame_rate
		self.frame_timer = Task(self.begin_frame, frame_time, repeat = True)
		self.average_frame_cpu_time = frame_time
	
	def open_app(self):
		dlog = OOBDialog()
		dlog.present()
		dlog.destroy()
	
	def open_cmd(self):
		if self.documents:
			self.documents[0].close_cmd()
		Application.open_cmd(self)

	def make_document(self, fref):
		return GameDoc()

	def make_window(self, doc):
		chip_win = ChipWindow(doc)
		chip_win.show()
		args = sys.argv
		if "-a" in args:
			doc.visit_any_level = True
		if "-w" not in args:
			doc.show_instructions()
		try:
			i = args.index("-l")
		except ValueError:
			pass
		else:
			n = int(args[i + 1])
			doc.select_level_number(n, force = True)

	def begin_frame(self):
		#print "application: begin_frame" ###
		#self.perform_frame_rate_regulation()
		for doc in self.documents:
			doc.begin_frame()
	
	def perform_frame_rate_regulation(self):
		last_cpu_time = self.last_cpu_time
		cpu_time_now = cpu_time()
		if last_cpu_time:
			cpu_time_last_used = cpu_time_now - last_cpu_time
			average_cpu_time = 0.9 * self.average_frame_cpu_time + 0.1 * cpu_time_last_used
			self.average_frame_cpu_time = average_cpu_time
			#print "CPU time used    =", cpu_time_last_used
			#print "Average cpu time =", average_cpu_time
			target_frame_time = 1.0 / frame_rate
			target_cpu_time = 0.9 * target_frame_time
			percent_busy = 100 * average_cpu_time / target_frame_time
			sampling_delayed = self.sample_delay > 0
			calculated_frame_time = average_cpu_time / 0.9
			calculated_frame_rate = max(min_frame_rate,
				min(max_frame_rate, 1.0 / calculated_frame_time))
			#print "Calculated frame time = %.2f" % calculated_frame_time
			#print "Calculated frame rate = %.2f" % calculated_frame_rate
			frame_rate_adjustment_indicated = not sampling_delayed \
				and (not 80 <= percent_busy <= 100) \
				and (frame_rate <> calculated_frame_rate)
			all_time_low = percent_busy > self.worst_percent_busy
			if profiling or sampling_delayed or frame_rate_adjustment_indicated or all_time_low:
				print "average cpu time =", average_cpu_time
				print "target cpu time  =", target_cpu_time
				print "%5.1f%% busy at %.2f fps" % (percent_busy, frame_rate)
			if all_time_low:
				self.worst_percent_busy = percent_busy
			if frame_rate_adjustment_indicated:
				self.adjust_frame_rate(calculated_frame_rate)
			if sampling_delayed:
				self.sample_delay -= 1
		self.last_cpu_time = cpu_time_now
	
	def adjust_frame_rate(self, new_frame_rate):
		global frame_rate
		frame_rate = new_frame_rate
		print "Adjusting frame rate to %.2f fps" % frame_rate
		self.init_frame_timer()
		self.worst_percent_busy = 0
		self.last_frame_begin_time = 0
		#self.sample_delay = 5

	def get_default_open_directory(self):
		return default_game_save_dir
