#--------------------------------------------------------------
#
#   SimChip2 - Waveforms
#
#--------------------------------------------------------------

from __future__ import division
from numpy import zeros, float32
from constants import \
	default_waveform_resolution as default_resolution, \
	default_wavespec_resolution, \
	default_waveform_time_range as default_time_range

#--------------------------------------------------------------

class Waveform(object):
	#  pin_index    int
	#  resolution   int                  time units per sample
	#  samples      array[num_samples]
	#  end          int                  number of valid samples
	
	end = 0
	
	def __init__(self, pin_index, resolution, num_samples, dtype = float32):
		self.pin_index = pin_index
		self.resolution = resolution
		self.samples = zeros([num_samples], dtype)
		self.end = 0
	
	def __len__(self):
		return len(self.samples)
	
	def sample_at_time(self, t):
		v = self.samples
		i = int(t / self.resolution)
		if 0 <= i < len(v):
			return v[i]
		else:
			return 0
	
	def set_sample_at_time(self, t, x):
		v = self.samples
		i = int(t / self.resolution)
		n = len(v)
		if 0 <= i < n:
			v[i] = x
			self.end = i + 1
	
	def set_samples_in_time_range(self, t1, t2, x):
		#print "waveforms: set_samples_in_time_range:", t1, t2, "to", x ###
		v = self.samples
		n = len(v)
		res = self.resolution
		i1 = max(0, int(t1 / res))
		i2 = min(n, int(t2 / res))
		#print "res =", res, "i1 =", i1, "i2 =", i2 ###
		v[i1 : i2] = x
		self.end = i2
	
	def set_from_spec(self, wave_spec, values):
		#print "waveforms: set_from_spec: pin", self.pin_index ###
		res = default_wavespec_resolution
		for i, c in enumerate(wave_spec):
			t = i * res
			v = values.get(c, 0)
			self.set_samples_in_time_range(t, t + res, v)
	
	def get_end_time(self):
		return self.end * self.resolution

#--------------------------------------------------------------

class WaveformSet(object):
	#  time_range   int
	#  tracks       [Waveform]
	#  track_map    {pin_index: Waveform}
	
	def __init__(self):
		self.time_range = default_time_range
		self.tracks = []
		self.track_map = {}
	
	def __len__(self):
		return len(self.tracks)
	
	def add_waveform_for_pin(self, i, spec = None, values = None, dtype = float32):
		res = default_resolution
		ns = self.time_range // res
		track = Waveform(i, res, ns, dtype)
		if spec:
			track.set_from_spec(spec, values)
		self.tracks.append(track)
		self.track_map[i] = track

	def track_for_pin(self, pin_index):
		return self.track_map.get(pin_index)
