#--------------------------------------------------------------
#
#   SimChip2 - Waveform view
#
#--------------------------------------------------------------

from __future__ import division
from numpy import array, arange, ndarray, float32
from GUI import ScrollableView, Font, StdColors
from constants import default_waveform_view_time_scale
import profile

profiling = 0

expectation_values = array([0, 1, 0, 1, 0.5], float32)

class WaveformView(ScrollableView):

	label_font = Font("Sans", 9)
	hmargin = 10
	vmargin = 10
	label_width = label_font.width("X" * 5) + 5
	track_height = 20
	track_spacing = 10
	time_scale = default_waveform_view_time_scale    # pixels per time unit
	voltage_range = 5.0
	wave_color = StdColors.black
	expectation_color = StdColors.grey
	cursor_color = StdColors.blue
	
	setup = property(lambda self: self.model.current_setup)
	waveforms = property(lambda self: self.model.current_waveforms)
	chip = property(lambda self: self.model.current_chip)

	def __init__(self, game, **kwds):
		ScrollableView.__init__(self, scrolling = 'hv', **kwds)
		self.model = game
		self.current_level_changed()
	
	def current_level_changed(self, *args):
		self.current_setup_changed()
	
	def current_setup_changed(self, *args):
		self.update_extent()
		self.invalidate()
	
	def update_extent(self):
		w = 2 * self.hmargin + self.label_width
		h = 2 * self.vmargin
		waves = self.waveforms
		if waves is not None:
			n = len(waves)
			if n > 0:
				w += int(waves.time_range * self.time_scale)
				h += n * self.track_height + (n - 1) * self.track_spacing
		#print "WaveformView: extent =", (w, h) ###
		self.extent = (w, h)
	
	def time_to_view_x(self, t):
		x0 = self.hmargin + self.label_width
		return x0 + t * self.time_scale

	def draw(self, canv, update_rect):
		if profiling:
			profile.begin("Draw waveforms")
		canv.erase_rect(update_rect)
		canv.font = self.label_font
		setup = self.setup
		waves = setup.waveforms
		exps = setup.expectations
		if waves is not None:
			h = self.track_height
			x = self.hmargin + self.label_width
			y = self.vmargin
			dy = h + self.track_spacing
			if waves is not None:
				for i, track in enumerate(waves.tracks):
					pin_setup = setup.pins[track.pin_index]
					self.draw_track_label(canv, y, pin_setup)
					exp = exps.track_for_pin(track.pin_index)
					if exp:
						self.draw_expectation(canv, x, y + h, y, exp)
					self.draw_track(canv, x, y + h, y, track, pin_setup.is_input())
					y += dy
			self.draw_cursor(canv)
		if profiling:
			profile.end("Draw waveforms")
	
	def draw_track_label(self, canv, y, pin_setup):
		canv.moveto(self.hmargin, y + self.track_height // 2 + canv.font.ascent // 2)
		canv.show_text(pin_setup.name)

	def draw_track(self, canv, x, y0, y1, track, square):
		#print "WaveformView.draw_track:", len(track), "samples" ###
		canv.pencolor = self.wave_color
		res = track.resolution
		samples = track.samples[:track.end]
		yrange = self.voltage_range
		self.draw_trace(canv, x, y0, y1, samples, res, yrange, square)
	
	def draw_expectation(self, canv, x, y0, y1, track):
		canv.pencolor = self.expectation_color
		trace = expectation_values[track.samples[:track.end]]
		self.draw_trace(canv, x, y0, y1, trace, track.resolution, 1.0, True)
	
	def draw_trace(self, canv, x, y0, y1, samples, res, yrange, square):
		if len(samples):
			s = self.time_scale * res # pixels per sample
			n = len(samples)
			yi = y0 + samples * ((y1 - y0) / yrange)
			xi = x + arange(n) * s
			if square:
				p = ndarray((2 * n, 2), float32)
				p[::2, 0] = xi
				p[1::2, 0] = xi + s
				p[::2, 1] = yi
				p[1::2, 1] = yi
			else:
				p = ndarray((n, 2), float32)
				p[:, 0] = xi
				p[:, 1] = yi
			canv.stroke_lines(p)

	def draw_cursor(self, canv):
		state = self.chip.state
		if state.auto_mode:
			#print "WaveformView.draw_cursor: t =", state.time
			x = self.time_to_view_x(state.time)
			#print "...x =", x ###
			h = self.extent[1]
			canv.pencolor = self.cursor_color
			canv.newpath()
			canv.moveto(x, 0)
			canv.lineto(x, h)
			canv.stroke()
