#--------------------------------------------------------------
#
#   SimChip2 - Level
#
#--------------------------------------------------------------

import os
from ConfigParser import SafeConfigParser
from html import HTMLPageSet
from chip import ChipParams, ChipSetup

#--------------------------------------------------------------

class XConfigParser(SafeConfigParser):

	def get(self, section, option, *args):
		if self.has_option(section, option):
			return SafeConfigParser.get(self, section, option, *args)
	
	def items(self, section, *args):
		if self.has_section(section):
			return SafeConfigParser.items(self, section, *args)
		else:
			return []

#--------------------------------------------------------------

class Level(object):
	#  parent    GameDoc
	#  key       string
	#  index     int
	#  title     string
	#  path      string
	#  html      HTMLPageSet   Level instructions
	#  unlocked  boolean
	#  no_test   boolean
	#  chip_params  ChipParams
	#  test_results  TestResulst
	
	game = property(lambda self: self.parent)
	current_doc_page_index = property(lambda self: self.doc_page)
	current_html = property(lambda self: self.html[self.doc_page])
	
#	doc_page = 0
	initially_unlocked = False
	test_results = None

	def __init__(self, parent, index, path):
		#print "Level %s: Loading %r" % (index, path) ###
		self.parent = parent
		self.index = index
		self.path = path
		inf_path = os.path.join(path, "level.inf")
		inf = XConfigParser()
		inf.read([inf_path])
		self.key = inf.get('Level', 'key')
		self.title = inf.get('Level', 'title')
		self.initially_unlocked = self.parse_boolean(inf, 'Level', 'unlocked')
		self.no_test = self.parse_boolean(inf, 'Level', 'no_test')
		self.parse_chip_params(inf)
		self.parse_html(inf)
	
	def parse_chip_params(self, inf):
		size = inf.get('Chip', 'size')
		num_pins = self.parse_int(inf, 'Chip', 'pins', 14)
		pin_specs = {}
		pins_by_name = {}
		for num, spec_str in inf.items('Pins'):
			self.parse_pin_spec(num, spec_str, pin_specs, pins_by_name)
		params = ChipParams(size, num_pins, pin_specs)
		params.speed = self.parse_int(inf, 'Chip', 'speed', 1)
		self.parse_wavespecs(inf, 'Waveforms', pins_by_name, params)
		self.chip_params = params
		self.test_setup = ChipSetup("Test", params, locked = True)
	
	def parse_int(self, inf, section, option, default):
		value = inf.get(section, option)
		if value:
			try:
				return int(value)
			except ValueError:
				print "Level %s: %s: %s: Bad int %r" % (self.key, value)
		else:
			return default
	
	def parse_boolean(self, inf, section, option):
		return inf.get(section, option) == "True"
	
	def parse_pin_spec(self, num, spec, pin_specs, pins_by_name):
		try:
			pin_index = int(num) - 1
			if "," not in spec:
				spec += ","
			kind, name = spec.split(",")
			pin_specs[pin_index] = (kind.strip(), name.strip())
			if name:
				pins_by_name[name.lower()] = pin_index
		except (ValueError, IndexError):
			print "Level %s: Bad pin spec: %s = %s", (self.key, num, spec)
	
	def parse_wavespecs(self, inf, section, pins_by_name, params):
		specs = params.waveforms
		tolerance = self.parse_int(inf, section, "*tolerance", 0)
		s1 = "0" + tolerance * "1"
		r1 = "0" + tolerance * "H"
		s2 = "1" + tolerance * "0"
		r2 = "1" + tolerance * "L"
		for name, spec in inf.items(section):
			if name == "*order":
				self.parse_waveorder(params, pins_by_name, spec)
			elif not name.startswith("*"):
				i = pins_by_name.get(name)
				if i is not None:
					spec = spec.replace(" ", "")
					if params.pin_specs[i][0] == 'out':
						spec = spec.replace(s1, r1).replace(s2, r2)
						#print "level.parse_wavespecs:", name, "-->", spec ###
					specs[i] = spec
	
	def parse_waveorder(self, params, pins_by_name, spec):
		result = []
		pool = params.wave_order
		for s in spec.split(","):
			name = s.strip().lower()
			i = pins_by_name.get(name)
			if i is not None:
				result.append(i)
				if i in pool:
					pool.remove(i)
		result.extend(pool)
		params.wave_order = result
	
	def parse_html(self, inf):
		html = HTMLPageSet()
		html_dir = os.path.join(self.path, "html")
		if os.path.isdir(html_dir):
			html_names = os.listdir(html_dir)
			html_names.sort()
			for name in html_names:
				if name.endswith(".html"):
					html_path = os.path.join(html_dir, name)
					html.parse_html_file(html_path)
		self.html = html
	
	def notify(self, *args):
		self.parent.notify(*args)
	
	def html_path_for_page_index(self, i):
		return os.path.join(self.path, "html", self.html[i])

	def set_test_results(self, results):
		self.test_results = results
		if results.passed():
			self.mark_completed()
			self.game.unlock_level_after(self)
		self.notify('test_results_changed')

	def description(self):
		return "%d - %s" % (self.index + 1, self.title)

	def mark_completed(self):
		cl = self.game.state.completed_levels
		if self.key not in cl:
			cl.add(self.key)
			self.game.changed()
	
	def mark_unlocked(self):
		ul = self.game.state.unlocked_levels
		if self.key not in ul:
			ul.add(self.key)
			self.game.changed()
	
	def is_completed(self):
		return self.key in self.game.state.completed_levels

	def is_unlocked(self):
		return self.key in self.game.state.unlocked_levels
