#--------------------------------------------------------------
#
#   SimChip2 - HTML processing
#
#--------------------------------------------------------------

import re, os
from BeautifulSoup import BeautifulSoup
from GUI import ScrollableView, Font, Image, rgb
from GUI.Geometry import offset_rect

#--------------------------------------------------------------

class HTMLPageSet(object):

	def __init__(self):
		self.pages = []
	
	def __len__(self):
		return len(self.pages)
	
	def __getitem__(self, i):
		return self.pages[i]
	
	def parse_html_file(self, path):
		f = open(path)
		tree = BeautifulSoup(f)
		f.close()
		self.pages.append(HTMLPage(path, tree))

#--------------------------------------------------------------

class HTMLPage(object):

	def __init__(self, path, tree):
		self.path = path
#		print "HTMLPage: path =", self.path ###
		self.tree = tree
		h1 = tree.h1
		if h1:
			self.title = h1.string
#			print "HTMLPage: title =", self.title ###

#--------------------------------------------------------------

galley_cache = {}

class HTMLView(ScrollableView):

	_debug_bg = True

	frame_color = rgb(0.5, 0.5, 0.5)
	watermark = Image.from_resource("images/watermark.png")

	tree = None
	galley = None

	def set_html(self, page):
		tree = page.tree if page else None
		if tree is not self.tree:
			self.tree = tree
			if tree:
				galley = galley_cache.get(tree)
				if not galley:
					galley = Galley(os.path.dirname(page.path))
					galley.format(tree.body)
					galley.flush_line()
					galley_cache[tree] = galley
				extent = galley.width, galley.height
			else:
				galley = None
				extent = (0, 0)
			self.galley = galley
			self.extent = extent
			self.invalidate()
	
	def draw(self, canv, update_rect):
		canv.erase_rect(self.viewed_rect())
		galley = self.galley
		if galley:
			canv.gsave()
			canv.pencolor = self.frame_color
			frame = (10, 10, galley.width - 10, galley.height - 10)
			canv.frame_rect(frame)
			wm = self.watermark
			src = wm.bounds
			dst = offset_rect(src, (frame[2] - wm.width, frame[1]))
			wm.draw(canv, src, dst)
			canv.grestore()
			x = galley.left_margin
			for line in galley.lines:
				canv.moveto(x, line.baseline)
				for run in line.runs:
#					print "HTMLView:", run.style, repr(run.text) ###
					style = run.style
					font = style.font
					text = run.text
					canv.font = font
					#print "HTMLView.draw: at", canv.current_point, repr(text) ###
					canv.show_text(text)
#					if 'underline' in style.options:
#						w = font.width(text)
#						ux, uy = canv.current_point
#						canv.fill_rect((ux - w, uy + 1, ux, uy + 2))
			for img in galley.images:
				image = img.image
				src = image.bounds
				dst = offset_rect(src, (img.left, img.top))
				image.draw(canv, src, dst)

#--------------------------------------------------------------

class Style(object):
	#  font         Font
	#  space_width  int
	#  ascent       int
	#  line_height  int
	family = None
	size = None
	options = set()
	is_block = False
	space_before = None
	space_after = None
	
	def __init__(self, base = None, options = None, **kwds):
		if base:
			self.__dict__.update(base.__dict__)
		if options:
			self.options = self.options | set(options)
		self.__dict__.update(kwds)
		family = self.family
		size = self.size
		if family and size:
			font = Font(self.family, self.size, self.options)
			self.font = font
			self.space_width = font.width(" ")
			self.ascent = font.ascent
			self.line_height = font.height #line_height
	
	def __str__(self):
		return "Style(%s,%s,%s,%s,before=%s,after=%s,line_height=%s)" % (
			self.family, self.size, self.options,
			"block" if self.is_block else "", self.space_before, self.space_after,
			self.line_height)
	
	def __ne__(self, other):
		return not (self.family == other.family and self.size == other.size
			and self.options == other.options)
	
	def modify(self, other):
		sb = self.space_before
		sa = self.space_after
		return Style(family = self.family or other.family,
			size = self.size or other.size,
			options = self.options | other.options,
			space_before = sb if sb is not None else other.space_before,
			space_after = sa if sa is not None else other.space_after)

#--------------------------------------------------------------

class Run(object):
	#  style   Style
	#  text    string
	
	def __init__(self, style, words):
		self.style = style
		self.text = "".join(words)

#--------------------------------------------------------------

class Line(object):
	#  baseline      int
	#  runs          [Run]
	
	def __init__(self, baseline, runs):
		self.baseline = baseline
		self.runs = runs

#--------------------------------------------------------------

class Img(object):
	#  left     int
	#  top      int
	#  image    Image
	
	def __init__(self, x, y, image):
		self.left = x
		self.top = y
		self.image = image

#--------------------------------------------------------------

class BlockStyle(Style):
	is_block = True

def HeadingStyle(space_before = 12, space_after = 12, **kwds):
	return BlockStyle(space_before = space_before,
		space_after = space_after, **kwds)

word_pat = re.compile(r"(\S+)|(\s+)")
default_style = Style(family = "Courier New", size = 14, space_before = 0, space_after = 0)

element_styles = dict(
	h1 = HeadingStyle(base = default_style, options = ['bold', 'underline'], space_before = 0),
	h2 = HeadingStyle(base = default_style, options = ['bold']),
	h3 = HeadingStyle(base = default_style),
	b = Style(options = ['bold']),
	i = Style(options = ['italic']),
	u = Style(options = ['underline']),
)

#print "html: default_style =", default_style
#print "h1 =", element_styles['h1']
#print "h2 =", element_styles['h2']
#print "h3 =", element_styles['h3']
#print "b =", element_styles['b']
#print "i =", element_styles['i']
#print "u =", element_styles['u']

#--------------------------------------------------------------

class Galley(object):
	#  top_margin         int
	#  bottom_margin      int
	#  min_height         int
	#  left_margin        int
	#  right_margin       int
	#  image_left_margin  int
	#  image_right_margin int
	#  lines              [Line]
	#  current_line       [Run]
	#  current_words      [string]
	#  current_style      Style
	#  last_block_style   Style
	#  x                  int
	#  y                  int
	#  width              int
	#  height             int
	#  images             [Img]
	#  doc_path           string
	
	def __init__(self, doc_path):
		self.top_margin = 20
		self.left_margin = 20
		self.right_margin = self.left_margin + 510
		self.image_left_margin = self.right_margin + 10
		self.image_right_margin = self.image_left_margin + 300
		self.width = self.image_right_margin + 20
		self.min_height = self.top_margin + 600 + 20
		self.height = 0
		self.lines = []
		self.current_line = []
		self.current_words = []
		self.current_style = default_style
		self.last_block_style = None
		self.x = self.left_margin
		self.y = self.top_margin
		self.images = []
		self.doc_path = doc_path
	
	def format(self, tree):
		tag = tree.name
		if tag == "br":
			if self.beginning_of_line():
				self.y += self.current_style.line_height
			else:
				self.flush_line()
		elif tag == 'img':
			self.leave_space_after_last_block()
			self.format_image(tree)
		else:
			style = element_styles.get(tag, default_style)
			#print "Galley:", tag, "style =", style ###
			block_style = None
			if style.is_block:
				block_style = style
				self.flush_line()
				last_block_style = self.last_block_style
				#print "Galley: last block style =", last_block_style, \
				#	"this block style =", block_style ###
				if last_block_style:
					self.y += max(style.space_before, last_block_style.space_after)
				else:
					self.y += style.space_before
			old_style = self.current_style
			new_style = style.modify(old_style)
			#print "Galley: new style =", new_style ###
			if new_style <> old_style:
				self.flush_words()
				self.current_style = new_style
			for elem in tree:
				if isinstance(elem, basestring):
					self.leave_space_after_last_block()
					#print "Galley: formatting", repr(elem)
					self.format_string(elem)
				else:
					self.format(elem)
			self.flush_words()
			#print "Galley: restoring style", old_style ###
			self.current_style = old_style
			if block_style:
				self.flush_line()
#				print "Galley: remembering last block style", block_style ###
				self.last_block_style = block_style
		self.height = max(self.y, self.min_height)
	
	def leave_space_after_last_block(self):
		last_block_style = self.last_block_style
		if last_block_style:
			#print "Galley: last block style =", last_block_style ###
			self.y += last_block_style.space_after
			#print "Galley: clearing last block style" ###
			self.last_block_style = None
	
	def format_image(self, elem):
		path = os.path.join(self.doc_path, elem['src'])
#		print "Galley: image:", repr(path)
		image = Image(path)
		x = self.image_left_margin
		w = image.width
		r =  self.image_right_margin
		if x + w > r:
			x = r - w
		img = Img(x, self.y, image)
		self.images.append(img)
	
	def format_string(self, s):
		style = self.current_style
		font = style.font
		right_margin = self.right_margin
		for match in word_pat.finditer(s):
			word = match.group().replace("&nbsp;", " ")
			if word.isspace():
				if self.beginning_of_line():
					continue
				word = " "
			w = font.width(word)
			#print "html: %r --> %s" % (word, w) ###
			if word <> " " and self.x + w > right_margin:
				self.flush_line()
			self.current_words.append(word)
			self.x += w
	
	def beginning_of_line(self):
		return not self.current_line and not self.current_words
	
	def flush_line(self):
		self.flush_words()
		runs = self.current_line
		if runs:
			#print "Galley.flush_line: x =", self.x ###
			ascent = 0
			line_height = 0
			for run in runs:
				style = run.style
				ascent = max(ascent, style.ascent)
				line_height = max(line_height, style.line_height)
			y = self.y
			self.lines.append(Line(y + ascent, runs))
			self.y = y + line_height
			self.x = self.left_margin
			self.current_line = []
	
	def flush_words(self):
		#print "Galley: flushing words" ###
		words = self.current_words
		if words:
			self.current_line.append(Run(self.current_style, words))
			self.current_words = []

#--------------------------------------------------------------

def dump_lines(lines):
	for line in lines:
		print "Line(baseline = %s)" % line.baseline
		for run in line.runs:
			print "  Run(style = %s)" % run.style
			print "    %r" % run.text
